<?php
namespace Barn2\Plugin\WC_Quick_View_Pro\Admin;

use Barn2\WQV_Lib\Registerable,
    Barn2\WQV_Lib\Util as Lib_Util,
    Barn2\WQV_Lib\Plugin\Licensed_Plugin,
    Barn2\Plugin\WC_Quick_View_Pro\Util,
    Barn2\WQV_Lib\WooCommerce\Admin\Custom_Settings_Fields,
    WC_Barn2_Plugin_Promo;

/**
 * Provides functions for the plugin settings page in the WordPress admin.
 *
 * Settings are registered under: WooCommerce -> Settings -> Products -> Quick view.
 *
 * @package   Barn2\woocommerce-quick-view-pro
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Settings_Page implements Registerable {

    const SECTION = 'quick-view-pro';

    private $plugin;

    public function __construct( Licensed_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function register() {
        $fields = new Custom_Settings_Fields();
        $fields->register();

        // Add sections & settings
        add_filter( 'woocommerce_get_sections_products', [ $this, 'add_section' ] );
        add_filter( 'woocommerce_get_settings_products', [ $this, 'add_settings' ], 10, 2 );

        // Save license setting
        $license_setting = $this->plugin->get_license_setting();
        add_filter( 'woocommerce_admin_settings_sanitize_option_' . $license_setting->get_license_setting_name(), [ $license_setting, 'save_license_key' ] );

        // Plugin promo
        $plugin_promo = new WC_Barn2_Plugin_Promo( $this->plugin->get_item_id(), $this->plugin->get_file(), self::SECTION );
        $plugin_promo->register();
    }

    public function add_section( $sections ) {
        $sections[self::SECTION] = __( 'Quick view', 'woocommerce-quick-view-pro' );
        return $sections;
    }

    public function add_settings( $settings, $current_section ) {
        // Check we're on the correct settings section
        if ( self::SECTION !== $current_section ) {
            return $settings;
        }

        $shop_page_url = get_permalink( wc_get_page_id( 'shop' ) );

        $plugin_settings = [
            [
                'id'    => 'quick_view_settings_start',
                'type'  => 'settings_start',
                'class' => 'quick-view-settings'
            ],
            // License key settings.
            [
                'title' => __( 'Quick View', 'woocommerce-quick-view-pro' ),
                'type'  => 'title',
                'id'    => 'quick_view_license_section',
                'desc'  => '<p>' . __( 'The following options control the WooCommerce Quick View Pro extension.', 'woocommerce-quick-view-pro' ) . '<p>'
                . '<p>'
                . Lib_Util::format_link( $this->plugin->get_documentation_url(), __( 'Documentation', 'woocommerce-quick-view-pro' ), true ) . ' | '
                . Lib_Util::format_link( $this->plugin->get_support_url(), __( 'Support', 'woocommerce-quick-view-pro' ), true )
                . '</p>'
            ],
            $this->plugin->get_license_setting()->get_license_key_setting(),
            $this->plugin->get_license_setting()->get_license_override_setting(),
            [
                'type' => 'sectionend',
                'id'   => 'quick_view_license_section'
            ],
            // Main quick view settings.
            [
                'title' => __( 'Options', 'woocommerce-quick-view-pro' ),
                'type'  => 'title',
                'id'    => 'quick_view_display_section',
            ],
            [
                'title'         => __( 'Shop display', 'woocommerce-quick-view-pro' ),
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[enable_button]',
                'desc'          => __( 'Show a Quick View button for each product', 'woocommerce-quick-view-pro' ),
                'default'       => 'yes',
                'checkboxgroup' => 'start'
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[enable_hover_button]',
                'desc'          => __( 'Display Quick View button on hover', 'woocommerce-quick-view-pro' ),
                'default'       => 'no',
                'checkboxgroup' => '',
                'desc_tip'      => '<em>' 
                    . __( 'May not work with all themes -', 'woocommerce-quick-view-pro' )
                    . sprintf( ' <a target="_blank" href="%s">', add_query_arg( '_b2-preview', 'qvp-hover', $shop_page_url ) )
                    . __( 'try it with yours', 'woocommerce-quick-view-pro' ) 
                    . ''
                    . '</a><span class="dashicons dashicons-external" style="vertical-align:middle;width:1.2em;height:1.2em;font-size:1.2em"></span></em>',
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[enable_product_link]',
                'desc'          => __( 'Open the Quick View by clicking the product name or image', 'woocommerce-quick-view-pro' ),
                'desc_tip'      => apply_filters( 'wc_quick_view_pro_settings_enable_product_link_description', '' ),
                'default'       => 'no',
                'checkboxgroup' => 'end',
                'desc_tip'      => '<em>' 
                    . __( 'May not work with all themes -', 'woocommerce-quick-view-pro' )
                    . sprintf( ' <a target="_blank" href="%s">', add_query_arg( '_b2-preview', 'qvp-full-click', $shop_page_url ) )
                    . __( 'try it with yours', 'woocommerce-quick-view-pro' ) 
                    . ''
                    . '</a><span class="dashicons dashicons-external" style="vertical-align:middle;width:1.2em;height:1.2em;font-size:1.2em"></span></em>',
            ],
            [
                'title'   => __( 'Button text', 'woocommerce-quick-view-pro' ),
                'type'    => 'text',
                'id'      => Util::OPTION_NAME . '[button_text]',
                'default' => __( 'Quick View', 'woocommerce-quick-view-pro' ),
                'css'     => 'width:160px'
            ],
            [
                'title'   => __( 'Button icon', 'woocommerce-quick-view-pro' ),
                'type'    => 'checkbox',
                'id'      => Util::OPTION_NAME . '[show_button_icon]',
                'desc'    => __( 'Show the button icon', 'woocommerce-quick-view-pro' ),
                'default' => 'yes'
            ],
            [
                'title'   => __( 'Quick View content', 'woocommerce-quick-view-pro' ),
                'type'    => 'select',
                'id'      => Util::OPTION_NAME . '[display_type]',
                'options' => array(
                    'image_only'   => __( 'Image only', 'woocommerce-quick-view-pro' ),
                    'details_only' => __( 'Product details only', 'woocommerce-quick-view-pro' ),
                    'both'         => __( 'Image and product details', 'woocommerce-quick-view-pro' ),
                ),
                'default' => 'both',
                'class'   => 'wc-enhanced-select'
            ],
            [
                'title'         => __( 'Product details', 'woocommerce-quick-view-pro' ),
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[show_reviews]',
                'desc'          => __( 'Show star ratings', 'woocommerce-quick-view-pro' ),
                'default'       => 'yes',
                'class'         => 'product-details',
                'checkboxgroup' => 'start'
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[tab_enable_reviews]',
                'desc'          => __( 'Show full reviews', 'woocommerce-quick-view-pro' ),
                'default'       => 'no',
                'checkboxgroup' => '',
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[show_price]',
                'desc'          => __( 'Show price', 'woocommerce-quick-view-pro' ),
                'default'       => 'yes',
                'checkboxgroup' => '',
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[show_description]',
                'desc'          => __( 'Show short description', 'woocommerce-quick-view-pro' ),
                'default'       => 'yes',
                'checkboxgroup' => ''
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[tab_enable_description]',
                'desc'          => __( 'Show full description', 'woocommerce-quick-view-pro' ),
                'default'       => 'no',
                'checkboxgroup' => '',
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[show_cart]',
                'desc'          => __( 'Show add to cart button', 'woocommerce-quick-view-pro' ),
                'default'       => 'yes',
                'checkboxgroup' => ''
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[show_meta]',
                'desc'          => __( 'Show meta information', 'woocommerce-quick-view-pro' ),
                'default'       => 'yes',
                'checkboxgroup' => ''
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[tab_enable_attrs]',
                'desc'          => __( 'Show attributes', 'woocommerce-quick-view-pro' ),
                'default'       => 'no',
                'checkboxgroup' => ''
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[show_details_link]',
                'desc'          => __( 'Show product details link', 'woocommerce-quick-view-pro' ),
                'default'       => 'no',
                'checkboxgroup' => 'end'
            ],
            [
                'title'         => __( 'Product image', 'woocommerce-quick-view-pro' ),
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[enable_zoom]',
                'desc'          => __( 'Enable image zoom', 'woocommerce-quick-view-pro' ),
                'default'       => 'yes',
                'class'         => 'product-image',
                'checkboxgroup' => 'start',
            ],
            [
                'type'          => 'checkbox',
                'id'            => Util::OPTION_NAME . '[enable_gallery]',
                'desc'          => __( 'Show gallery thumbnails', 'woocommerce-quick-view-pro' ),
                'default'       => 'yes',
                'checkboxgroup' => 'end'
            ],
            [
                'title'   => __( 'Gallery thumbnail style', 'woocommerce-quick-view-pro' ),
                'type'    => 'select',
                'id'      => Util::OPTION_NAME . '[gallery_style]',
                'options' => array(
                    'thumbs'  => __( 'Thumbnails', 'woocommerce-quick-view-pro' ),
                    'bullets' => __( 'Bullets', 'woocommerce-quick-view-pro' ),
                ),
                'default' => 'thumbs',
                'css'     => 'width:150px;',
                'class'   => 'wc-enhanced-select'
            ],
            [
                'type' => 'sectionend',
                'id'   => 'quick_view_display_section'
            ],
            [
                'id'   => 'quick_view_settings_end',
                'type' => 'settings_end'
            ]
        ];

        return $plugin_settings;
    }

}
